﻿using System;
using System.Collections.Generic;
using System.Configuration;
using System.Data;
using System.Data.SqlClient;
using System.Linq;
using System.Text;

internal class PatientImageRepository
{
    private const int TOKEN_REFRESH_TIME_HOURS = 8;

    private const string CONNECT_STRING_NAME = "RegistryConnectionString";

    private const string USP_INSERT_PATIENT_IMAGES = "[EFR].[usp_InsertPatientImages]";

    private const string SP_GET_IMAGE_ANALYSIS = "EFR.usp_GetImageAnalysisForPatient";
    private const string SP_GET_PATIENT_IMAGE_BODY_PARTS = "[EFR].[usp_GetPatientImageBodyParts]";
    private const string SP_GET_STANDARD_IMAGE_FRAGMENT_SIZES = "EFR.usp_GetStandardImageFragmentSizes";

    private const string SP_INSERT_UPDATE_IMAGE_ANALYSIS = "EFR.usp_InsertUpdateImageAnalysisForPatient";
    private const string SP_IMAGE_ANALYSIS_HISTORY_EXISTS = "[EFR].[usp_ImageAnalysisHistoryExists]";

    private const string SP_DELETE_IMAGE_ANALYSIS_FOR_PATIENT = "[EFR].[usp_DeleteImageAnalysisForPatient]";
    private const string SP_REVERT_IMAGE_ANALYSIS_FOR_PATIENT = "[EFR].[usp_RevertImageAnalysisForPatient]";

    private const string SP_GET_PATIENT_IMAGE_SOURCE_REFRESH_DATE = "[EFR].[usp_GetPatientImageSourceRefreshDate]";
    private const string SP_GET_IMAGES_FOR_PATIENT = "[EFR].[usp_GetImagesForPatient]";

    internal static IEnumerable<string> LoadReferenceImageFragmentSizes()
    {
        return ImagingDataRepository.LoadReferenceValuesNameWithDescription(SP_GET_STANDARD_IMAGE_FRAGMENT_SIZES);
    }

    internal PatientImageAnalysis GetPatientImageAnalysis(string patientIcn)
    {
        using (var imageAnalysisTable = SqlProvider.ExecuteSPDataTable(
            CONNECT_STRING_NAME,
            SP_GET_IMAGE_ANALYSIS,
            new[] { patientIcn }))
        {
            if (imageAnalysisTable != null
                && imageAnalysisTable.Rows != null
                && imageAnalysisTable.Rows.Count > 0)
            {
                var imageAnalysisRow = imageAnalysisTable.Rows[0];

                var result = new PatientImageAnalysis
                (
                    imageAnalysisRow.Field<bool?>("IMAGES_IN_CPRS_FLAG"),
                    imageAnalysisRow.Field<bool?>("FRAGMENT_VERIFIED_FLAG"),
                    imageAnalysisRow.Field<bool?>("BONE_FRACTURE_DOCUMENTED_FLAG"),
                    imageAnalysisRow.Field<bool?>("FRAGMENTS_IN_JOIN_SPACE_FLAG"),
                    imageAnalysisRow.Field<int?>("STD_IMAGE_FRAGMENT_SIZE_ID"),
                    imageAnalysisRow.Field<string>("JOINT_SPACE"),
                    imageAnalysisRow.Field<string>("OTHER_OBSERVATIONS"),
                    imageAnalysisRow.Field<DateTime?>("DATE_REVIEWED")
                );

                return result;
            }
        }

        return null;
    }

    internal string GetBodyPartsImaged(int patientImageId)
    {
        using (var imagedBodyPartsTable = SqlProvider.ExecuteSPDataTable(
            CONNECT_STRING_NAME,
            SP_GET_PATIENT_IMAGE_BODY_PARTS,
            new[] { string.Format("{0}", patientImageId) }))
        {
            if (imagedBodyPartsTable != null
                && imagedBodyPartsTable.Rows != null
                && imagedBodyPartsTable.Rows.Count > 0)
            {
                StringBuilder builder = new StringBuilder();

                foreach (DataRow row in imagedBodyPartsTable.Rows)
                {
                    if (builder.Length > 0) builder.Append(", ");

                    builder.Append(row.Field<string>("DESCRIPTION_TEXT"));
                }

                return builder.ToString();
            }
        }

        return null;
    }

    internal void InsertUpdateStudies(IReadOnlyList<VixStudy> studies)
    {
        var connectionString = ConfigurationManager.ConnectionStrings[CONNECT_STRING_NAME].ConnectionString;

        if (studies != null && studies.Any())
        {
            using (SqlConnection connection = new SqlConnection(connectionString))
            using (SqlCommand updateCommand = new SqlCommand(USP_INSERT_PATIENT_IMAGES, connection))
            {
                connection.Open();

                updateCommand.CommandType = CommandType.StoredProcedure;

                SqlParameter tvpStudies = updateCommand.Parameters.AddWithValue("@Studies", GetInsertUpdateStudiesParameterTable(studies));
                tvpStudies.SqlDbType = SqlDbType.Structured;

                updateCommand.ExecuteNonQuery();
            }
        }
    }

    internal static DataTable GetInsertUpdateStudiesParameterTable(IEnumerable<VixStudy> collection)
    {
        var parameters = new DataTable();
        parameters.Columns.Add("StudyId");
        parameters.Columns.Add("PatientId");
        parameters.Columns.Add("StudyDate");
        parameters.Columns.Add("Description");
        parameters.Columns.Add("ViewerUrl");

        foreach (var study in collection)
        {
            DataRow row = parameters.Rows.Add();
            row.SetField("StudyId", study.StudyId);
            row.SetField("PatientId", study.PatientId);
            row.SetField("StudyDate", study.StudyDate);
            row.SetField("Description", study.StudyDescription);
            row.SetField("ViewerUrl", study.ViewerUrl);
        }

        return parameters;
    }

    internal int DeleteImageAnalysis(string patientIcn)
    {
        return SqlProvider.PerformUpdate(
            CONNECT_STRING_NAME,
            SP_DELETE_IMAGE_ANALYSIS_FOR_PATIENT,
            new[] { string.Format("{0}", patientIcn) });
    }

    internal int InsertUpdateImageAnalysis(object[] parameters)
    {
        return SqlProvider.PerformUpdate(
            CONNECT_STRING_NAME,
            SP_INSERT_UPDATE_IMAGE_ANALYSIS,
            parameters);
    }

    internal bool ImageAnalysisHistoryExists(string patientIcn)
    {
        using (var checkTable = SqlProvider.ExecuteSPDataTable(
            CONNECT_STRING_NAME,
            SP_IMAGE_ANALYSIS_HISTORY_EXISTS,
            new[] { string.Format("{0}", patientIcn) }))
        {
            if (checkTable != null
                && checkTable.Rows != null
                && checkTable.Rows.Count > 0)
            {
                return checkTable.Rows[0].Field<bool>("ExistsFlag");
            }
        }

        return false;
    }

    internal DataTable GetPatientImagesDataTable(string patientIcn)
    {
        var result = SqlProvider.ExecuteSPDataTable(
            CONNECT_STRING_NAME,
            SP_GET_IMAGES_FOR_PATIENT,
            new[] { string.Format("{0}", patientIcn) });

        return result;
    }

    internal bool ShouldRefreshImageData(int patientId)
    {
        DateTime? refreshDate = null;

        using (var checkTable = SqlProvider.ExecuteSPDataTable(
            CONNECT_STRING_NAME,
            SP_GET_PATIENT_IMAGE_SOURCE_REFRESH_DATE,
            new[] { string.Format("{0}", patientId) }))
        {
            if (checkTable != null
                && checkTable.Rows != null
                && checkTable.Rows.Count > 0)
            {
                refreshDate = checkTable.Rows[0].Field<DateTime?>("SourceRefreshDate");
            }
        }

        return !refreshDate.HasValue
            || refreshDate.Value.Date != DateTime.Now.Date
            || refreshDate.Value <= DateTime.Now.AddHours(-TOKEN_REFRESH_TIME_HOURS);
    }
}

